/*
 * p/n:
 * Program name:         Dump Formatter formatting routines
 * Author:               Pete Madderom
 * Backup:
 *
 * Description:
 *    This component contains a series of routines to produce
 *    formatted print-outs of the various kernel structures.
 *
 *    All are called with one parameter giving the memory address
 *    of the table in the dumped system.  Each routine is responsible
 *    for obtaining the memory to be printed by calling getblk.
 */

#include "/usr/src/sys/h/param.h"
#include "/usr/src/sys/h/proc.h"
#include "/usr/src/sys/h/tty.h"
#include "/usr/src/sys/h/dir.h"
#include "/usr/src/sys/h/user.h"
#include "/usr/src/sys/h/coretab.h"
#include "/usr/src/sys/h/inode.h"
#include "/usr/src/sys/h/io.h"
#include "/usr/src/sys/h/ios.h"
#include "/usr/src/sys/h/text.h"
#include "/usr/src/sys/h/filsys.h"
#include "/usr/src/sys/h/ino.h"
#include "/usr/src/sys/h/buf.h"
#include "/usr/src/sys/h/ioconf.h"
#include "/usr/src/sys/h/dasd.h"
#include "/usr/src/sys/h/tube.h"
#include "/usr/src/sys/h/file.h"
#include "/usr/src/sys/h/mount.h"
#include "/usr/src/sys/h/pagetab.h"
#include "dump.h"
#include <stdio.h>

extern int getbytes(), error(), symlook();

extern FILE *out;
/*
 * routine to print contents of buf structure
 */
prtbuf(loc)
int loc;
{
        struct buf *p;

        fprintf(out, "\nBuf structure at address %8x\n  ",loc);
        p = getblk(loc, sizeof(struct buf));
        if (p == -1) error(1, "illegal address");

        fprintf(out, " flags: ");
        if (p->b_flags & B_READ)   fprintf(out, "READ ");
        if (p->b_flags & B_DONE)   fprintf(out, "DONE ");
        if (p->b_flags & B_ERROR)  fprintf(out, "ERROR ");
        if (p->b_flags & B_BUSY)   fprintf(out, "BUSY ");
        if (p->b_flags & B_PHYS)   fprintf(out, "PHYS ");
        if (p->b_flags & B_MAP)    fprintf(out, "MAP ");
        if (p->b_flags & B_WANTED) fprintf(out, "WANTED ");
        if (p->b_flags & B_AGE)    fprintf(out, "AGE ");
        if (p->b_flags & B_ASYNC)  fprintf(out, "ASYNC ");
        if (p->b_flags & B_DELWRI) fprintf(out, "DELWRI ");
        if (p->b_flags & B_TAPE)   fprintf(out, "TAPE ");
        if (p->b_flags & B_PBUSY)  fprintf(out, "PBUSY ");
        if (p->b_flags & B_PACK)   fprintf(out, "PACK ");
        fprintf(out, "\n  ");

        fprintf(out, " forw->%8x", p->b_forw);
        fprintf(out, "  back->%8x\n  ", p->b_back);

        fprintf(out, " avail forw->%8x", p->av_forw);
        fprintf(out, "  back->%8x\n  ", p->av_back);

        fprintf(out, " dev: %d/%d",major(p->b_dev),minor(p->b_dev));
        fprintf(out, "  bcount: %d", (int)p->b_bcount);
        fprintf(out, "  union: %8x\n  ", (int)p->b_un.b_addr);

        fprintf(out, " blkno: %d", p->b_blkno);
        fprintf(out, "  error: %2x", (int)p->b_error);
        fprintf(out, "  resid: %d\n", (int)p->b_resid);
}

/*
 *  routine to print contents of coretab structure
 *  (Since there is only one coretab, this routine ignores the
 *  parameter passed and obtains various symbols for itself.)
 */
prtcrtab()
{
        struct coretab *temptab;

        struct pageinfo {
               char p_key;
               char p_used;
        } *ptrpage, *ptrpagei;

        unsigned numpage;

        int address, *bufi, coretab, count, ffpage, i, pagecnt;

        /*
         *  Get the coretab address and value
         */
        address = symlook("coretab");
        if (address == -1)
                error(1, "coretab not found");
        bufi = getblk(address, sizeof(int));
        if (bufi == -1)
                error(1, "illegal coretab address:%8x", address);
        coretab = *bufi;

        fprintf(out, "\nCoretab structure at address %8x\n  ", coretab);

        /*
         *  Get the size of the machine
         */
        address = symlook("memlim");
        if (address == -1)
                error(1, "memlim not found");
        bufi = getblk(address, sizeof(int));
        if (bufi == -1)
                error(1, "illegal memlim address");
        numpage = pa2pi(*bufi) + 1;          /* total number of pages */

        /*
         *  Produce an allocation map
         */
        ptrpage = ptrpagei = calloc(numpage, sizeof(struct pageinfo));
        if (ptrpage == NULL)
                error(1, "could not allocate buffer");

        /*
         *  First set all pages to allocated
         */
        address = coretab;
        for (i = 0; i < numpage; i++) {
             ptrpage->p_used = '*';
             temptab = getblk(address, sizeof(struct coretab));
             if (temptab  == -1)
                     error(1, "could not locate coretab[%d]",i);
             ptrpage->p_key = temptab->c_storkey;     /* save storage key */
             address += sizeof(struct coretab);
             ptrpage++;
        }

        /*
         *  Now go through the free list
         */
        address = symlook("ffpage");
        if (address == -1)
                error(1, "ffpage not found");
        bufi = getblk(address, sizeof(int));
        if (bufi == -1)
                error(1, "illegal ffpage address");
        ffpage = *bufi;        /* get first free page index */

        fprintf(out, " ffpage: %3x\n   storage keys of pages (*'s -> allocated)",
                      ffpage);

        count = 0;
        while (ffpage != NULL) {
              ptrpage = ptrpagei + ffpage;
              ptrpage->p_used = ' ';
              address = coretab + ffpage*sizeof(struct coretab);
              temptab = getblk(address, sizeof(struct coretab));
              if (temptab == -1)
                      error(1, "could not locate coretab[%d]", i);
              count++;
              ffpage = temptab->c_nxtfpag;
         }

        /*
         *  Determine page count
         */
        address = symlook("pagecnt");
        if (address == -1)
                error(1, "pagecnt not found");
        bufi = getblk(address, sizeof(int));
        if (bufi == -1)
                error(1, "illegal pagecnt address");
        pagecnt = *bufi;             /* get pagecnt value */

        /*
         *  Finally print out coretab structure
         */
        ptrpage = ptrpagei;
        for (i = 0; i < numpage; i++) {
                if (i%16 == 0)
                        fprintf(out, "\n   %3x", i);
                fprintf(out, " %2x%c", ptrpage->p_key, ptrpage->p_used);
                ptrpage++;
        }

        fprintf(out, "\n   pagecnt: %d", pagecnt);
        if (count != pagecnt)
                fprintf(out, "  actual count: %d", count);
        fprintf(out, "\n");

        /*
         * Free the buffer
         */
        cfree(ptrpagei);
}

/*
 * routine to print contents of inode structure
 */

prtinode(loc)
int loc;
{
        struct inode *p;

        int i, type;

        fprintf(out, "\nInode structure at address %8x\n  ",loc);
        p = getblk(loc, sizeof(struct inode));
        if (p == -1)
                error(1, "illegal address");

        fprintf(out, " flag: ");
        if (p->i_flag & ILOCK)   fprintf(out, "LOCK ");
        if (p->i_flag & IUPD)    fprintf(out, "UPD ");
        if (p->i_flag & IACC)    fprintf(out, "ACC ");
        if (p->i_flag & IMOUNT)  fprintf(out, "MOUNT ");
        if (p->i_flag & IWANT)   fprintf(out, "WANT ");
        if (p->i_flag & ITEXT)   fprintf(out, "TEXT ");
        if (p->i_flag & ICHG)    fprintf(out, "CHG ");

        fprintf(out, " count: %d", (int)p->i_count);
        fprintf(out, "  dev: %d/%d",major(p->i_dev), minor(p->i_dev));
        fprintf(out, "  number: %d\n  ", (int)p->i_number);

        fprintf(out, " mode: %4x", (int)p->i_mode);
        fprintf(out, "  nlink: %d", (int)p->i_nlink);
        fprintf(out, "  uid: %d", (int)p->i_uid);
        fprintf(out, "  gid: %d", (int)p->i_gid);
        fprintf(out, "  size: %d\n  ", p->i_size);

        type = p->i_mode & IFMT;       /* get file type */
        if (type == IFMPC || type == IFMPB) {
               fprintf(out, " rdev: %4x\n  ", p->i_un.i_rdev);

               fprintf(out, " state: %4x", (int)p->i_un.i_group.g_state);
               fprintf(out, "  index: %d", (int)p->i_un.i_group.g_index);
               fprintf(out, "  rot: %2x\n  ", (int)p->i_un.i_group.g_rot);

               fprintf(out, " group->%8x", p->i_un.i_group.g_group);
               fprintf(out, "  inode->%8x", p->i_un.i_group.g_inode);
               fprintf(out, "  file->%8x\n  ", p->i_un.i_group.g_file);

               fprintf(out, " rotmask: %4x", p->i_un.i_group.g_rotmask);
               fprintf(out, "  datq: %d\n   group channel pointers\n   ",
                          p->i_un.i_group.g_datq);
               for (i = 0; i < 8; i++)
                   fprintf(out, "%8x ", p->i_un.i_group.g_chans[i]);
               fprintf(out, "\n   ");
               for (i = 8; i < NINDEX; i++)
                       fprintf(out, "%8x ", p->i_un.i_group.g_chans[i]);
               fprintf(out, "\n  ");
        } else {
               fprintf(out, " addr: ");
               for (i = 0; i < 6; i++)
                       fprintf(out, "%d ", p->i_un.i_addr[i]);
               fprintf(out, "\n         ");
               for (i = 6; i < NADDR; i++)
                       fprintf(out, "%d ", p->i_un.i_addr[i]);
               fprintf(out, "\n   lastr: %d\n  ", p->i_un.i_lastr);
        }
}

/*
 * routine to print contents of a channel structure
 */
prtchan(loc)
int loc;
{
        struct chan *p;
        int i;

        fprintf(out, "\nChan structure at address %8x", loc);
        if (io_chan != -1)
                fprintf(out, " for channel %x", io_chan);
        fprintf(out, "\n   ");

        p = getblk(loc, sizeof(struct chan));
        if (p == -1)
                error(1, "illegal address");

        fprintf(out, " control unit pointers\n   ");
        for (i = 0; i < 8; i++)
               fprintf(out, "%8x ", p->ch_cu[i]);
        fprintf(out, "\n   ");
        for (i = 8; i < 16; i++)
               fprintf(out, "%8x ", p->ch_cu[i]);

        fprintf(out, "\n   type: ");
        switch(p->ch_type) {
        case SELECTOR:
            fprintf(out, "SELECTOR"); break;
        case MX:
            fprintf(out, "MX"); break;
        case BMX:
            fprintf(out, "BMX"); break;
        default:
            fprintf(out, "%d", p->ch_type);
        }
        fprintf(out, "  state: ");
        switch(p->ch_state) {
        case FREE:
            fprintf(out, "FREE"); break;
        case WORKING:
            fprintf(out, "WORKING"); break;
        case HUNG:
            fprintf(out, "HUNG"); break;
        case DEAD:
            fprintf(out, "DEAD"); break;
        default:
            fprintf(out, "%d", p->ch_state);
        }
        fprintf(out, "  q->%8x\n",p->ch_q);
}

/*
 * routine to print contents of a control unit structure
 */
prtcu(loc)
int loc;
{
        struct cu *p;
        int i;

        fprintf(out, "\nCu structure at address %8x",loc);
        if (io_cu != -1)
                fprintf(out, " for control unit %2x", io_cu);
        fprintf(out, "\n  ");

        p = getblk(loc,sizeof(struct cu));
        if (p == -1)
               error(1, "illegal address");

        fprintf(out, " unit pointers\n   ");
        for (i = 0; i < 8; i++)
               fprintf(out, "%8x ", p->cu_un[i]);
        fprintf(out, "\n   ");
        for (i = 8; i < 16; i++)
               fprintf(out, "%8x ", p->cu_un[i]);

        fprintf(out, "\n   state: ");
        switch(p->cu_state) {
        case FREE:
            fprintf(out, "FREE"); break;
        case WORKING:
            fprintf(out, "WORKING"); break;
        case HUNG:
            fprintf(out, "HUNG"); break;
        case DEAD:
            fprintf(out, "DEAD"); break;
        default:
            fprintf(out, "%d", p->cu_state);
        }
        fprintf(out, " q->%8x\n", p->cu_q);
}

/*
 * routine to print contents of a unit structure
 */
prtunit(loc)
int loc;
{
        struct unit *p;
        int i;

        fprintf(out, "\nUnit structure at address %8x",loc);
        if (io_unit != -1)
                fprintf(out, " for unit %3x", io_unit);
        fprintf(out, "\n  ");

        p = getblk(loc, sizeof(struct unit));
        if (p == -1)
                error(1, "illegal address");

        fprintf(out, " sense ccw: %16X\n  ", p->un_sccw);

        fprintf(out, " state: ");
        switch(p->un_state) {
        case FREE:
            fprintf(out, "FREE"); break;
        case WORKING:
            fprintf(out, "WORKING"); break;
        case HUNG:
            fprintf(out, "HUNG"); break;
        case DEAD:
            fprintf(out, "DEAD"); break;
        default:
            fprintf(out, "%d", p->un_state);
        }
        fprintf(out, "  active q->%8x", p->un_actv);
        fprintf(out, " q->%8x\n  ", p->un_q);

        fprintf(out, " asynch address->%8x", p->un_intr);
        fprintf(out, "  arg: %8x", p->un_arg);
        fprintf(out, "  mint: %d\n  ", p->un_mint);

        fprintf(out, " csw: %8x %8x sense info: ", p->un_csw.cs_word1, p->un_csw.cs_word2);
        for (i = 0; i < SENSEL; i++)
               fprintf(out, "%2x", (int)p->un_sense[i]);
        fprintf(out, "\n");
}

/*
 * Routine to print contents of an I/O queue.  Puts out top heading
 * indicating whether for chan, cu or unit.
 */
prtioq(loc)
int loc;
{
        if (io_chan != -1)
                if (io_cu != -1)
                        if (io_unit != -1)
                                fprintf(out, "\nIoq entries for unit %3x\n",
                                        io_unit);
                        else
                                fprintf(out, "\nIoq entries for control unit %2x\n",
                                        io_cu);
                 else
                        fprintf(out, "\nIoq entries for channel %x\n",
                               io_chan);
        else    fprintf(out, "\n");

        printioq(loc);
}

/*
 * Routine to print contents of an active ioq. Puts out heading, then
 * merges with routine for waiting queues.
 */
prtactive(loc)
int loc;
{
        fprintf(out, "\nActive ioq entries");
        if (io_unit != -1)
                fprintf(out, " for unit %3x", io_unit);
        fprintf(out, "\n");

        printioq(loc);
}

/*
 * Internal routine to print all elements of an I/O queue.
 */
printioq(loc)
int loc;
{
        struct ioq *p;

        while (loc != 0) {
                p = getblk(loc, sizeof(struct ioq));
                if (p == -1) {
                         error(1, "illegal address");
                         return(0);
                }
                fprintf(out, "Ioq element at address %8x\n",loc);
                prtioqe(p);
                loc = p -> io_next;
        }
}

/*
 *  Routine to print a single IOQ element (storage block already
 *  obtained).
 */
prtioqe(p)
struct ioq      *p;
{
        fprintf(out, "   next->%8x\n", p->io_next);

        fprintf(out, "   qnum: %d", p->io_qnum);
        fprintf(out, "   type: ");
        switch(p->io_type) {
        case SIO:
            fprintf(out, "SIO"); break;
        case TIO:
            fprintf(out, "TIO"); break;
        case HIO:
            fprintf(out, "HIO"); break;
        case CLR_HDV:
            fprintf(out, "CLR_HDV"); break;
        case CLR_CLRIO:
            fprintf(out, "CLR_CLRIO"); break;
        default:
            fprintf(out, "%d", p->io_type);
        }
        fprintf(out, "   caw->%8x\n", p->io_caw);

        fprintf(out, "   devaddr: %4x\n", p->io_devaddr);

        fprintf(out, "   intr->%8x", p->io_intr);
        fprintf(out, "  arg: %8x", p->io_arg);
        fprintf(out, "  mint: %d\n", p->io_mint);
}

/*
 * routine to print contents of a proc structure
 */
prtproc(loc)
int loc;
{
        struct proc *ptr;

        int i, n, p;

        fprintf(out, "\nProcess structure at address %8x", loc);
        ptr = getblk(loc, sizeof(struct proc));
        if (ptr == -1)
                error(1,"illegal address");

        fprintf(out, " for process %d\n  ", (int)ptr->p_pid);

        fprintf(out, " stat: ");
        switch(ptr->p_stat) {
        case SSLEEP:
            fprintf(out, "SLEEP");
            break;
        case SRUN:
            fprintf(out, "RUN");
            break;
        case SIDL:
            fprintf(out, "IDL");
            break;
        case SZOMB:
            fprintf(out, "ZOMB");
            break;
        case SSTOP:
            fprintf(out, "STOP");
            break;
        default:
            fprintf(out, "%d", (int)ptr->p_stat);
            break;
        }
        fprintf(out, "  flag:");
        if (ptr->p_flag & SLOAD) fprintf(out, " LOAD");
        if (ptr->p_flag & SSYS)  fprintf(out, " SYS");
        if (ptr->p_flag & SLOCK) fprintf(out, " LOCK");
        if (ptr->p_flag & SSWAP) fprintf(out, " SWAP");
        if (ptr->p_flag & STRC)  fprintf(out, " TRC");
        if (ptr->p_flag & SWTED) fprintf(out, " WTED");
        if (ptr->p_flag & SULOCK) fprintf(out, " ULOCK");
        fprintf(out, " (%2x)\n  ", (int)ptr->p_flag);

        fprintf(out, " prio: %d", (int)ptr->p_pri);
        fprintf(out, "  cpu: %d", (int)ptr->p_cpu);
        fprintf(out, "  nice: %d\n  ", (int)ptr->p_nice);

        fprintf(out, " signals pending:");
        n = (NSIG+8)/8;
        p = 1;
        for ( i = 0; i < n; i ++)
             if (ptr->p_sig[i]) {
                p = 0;            /* don't need none message */
                fprintf(out, "%d", (int)ptr->p_sig[i]);
             }
        if (p) fprintf(out, " none");
        fprintf(out, "  issig: %d\n  ", (int)ptr->p_issig);

        fprintf(out, " time: %ld\n  ", ptr->p_time>>12);

        fprintf(out, " uid: %d", (int)ptr->p_uid);
        fprintf(out, "  pgrp: %d", (int)ptr->p_pgrp);
        fprintf(out, "  pid: %d", (int)ptr->p_pid);
        fprintf(out, "  ppid: %d\n  ", (int)ptr->p_ppid);

        fprintf(out, " user->%8x", ptr->p_addr);
        fprintf(out, "  wchan->%8x", ptr->p_wchan);
        fprintf(out, "  text->%8x", ptr->p_textp);
        fprintf(out, "  proc->%8x\n  ", ptr->p_link);

        fprintf(out, " clktime: %d\n",ptr->p_clktim);

        /*
         * print the associated command.  Must be last, since it
         * causes the buffer to be reused.
         */
        fprintf(out, "   command:");
        printcmd(ptr);

}

/*
 * routine to print the contents of a text structure
 */
prttext(loc)
int loc;
{
        struct text *p;
        struct pagetab *tseg, *dseg;

        fprintf(out, "\nText structure at address %8x\n  ",loc);
        p =getblk(loc, sizeof(struct text));
        if (p == -1)
                error(1, "illegal address");

        fprintf(out, " tseg->%8x/", p->x_tseg.u_pages);
        fprintf(out, "%d\n  ", (int)p->x_tseg.u_size);

        fprintf(out, " dseg->%8x/", p->x_dseg.u_pages);
        fprintf(out, "%d\n  ", (int)p->x_dseg.u_size);

        fprintf(out, " iptr->%8x\n   header information\n  ", p->x_iptr);

        fprintf(out, " magic: %d", p->x_exdata.ux_mag);
        fprintf(out, "  tsize: %d", (int)p->x_exdata.ux_tsize);
        fprintf(out, "  dsize: %d", (int)p->x_exdata.ux_dsize);
        fprintf(out, "  bsize: %d\n  ", (int)p->x_exdata.ux_bsize);

        fprintf(out, " ssize: %d", (int)p->x_exdata.ux_ssize);
        fprintf(out, "  entloc: %8x", (int)p->x_exdata.ux_entloc);
        fprintf(out, "  unused: %d", (int)p->x_exdata.ux_unused);
        fprintf(out, "  relflg: %d\n  ", (int)p->x_exdata.ux_relflg);

        fprintf(out, " all in: %d", p->x_allin);
        fprintf(out, "  count: %d", p->x_count);
        fprintf(out, "  flag: %2x\n  ", p->x_flag);
        /*
         * now print contents of page tables
         */
        tseg = p->x_tseg.u_pages;
        dseg = p->x_dseg.u_pages;
        if (tseg != NULL) {
             fprintf(out, " text page tables\n  ");
             prtpagetab(tseg);
        }
        if (dseg != NULL) {
             fprintf(out, " data page tables\n  ");
             prtpagetab(dseg);
        }
}

/*
 * routine to print the contents of a tty structure
 */
prttty(loc)
int loc;
{
        struct tty *p;

        fprintf(out, "\nTty structure at address %8x",loc);
        p = getblk(loc, sizeof(struct tty));
        if (p == -1)
                error(1, "illegal address");
        fprintf(out, " for tty %4x dev %d/%d\n  ", (int)p->t_addr, major(p->t_dev), minor(p->t_dev));

        fprintf(out, " rawq count: %d", p->t_rawq.c_cc);
        fprintf(out, "  first char->%8x", p->t_rawq.c_cf);
        fprintf(out, "  last char->%8x\n  ", p->t_rawq.c_cl);

        fprintf(out, " canq count: %d", p->t_canq.c_cc);
        fprintf(out, "  first char->%8x", p->t_canq.c_cf);
        fprintf(out, "  last char->%8x\n  ", p->t_canq.c_cl);

        fprintf(out, " outq count: %d", p->t_outq.c_cc);
        fprintf(out, "  first char->%8x", p->t_outq.c_cf);
        fprintf(out, "  last char->%8x\n  ", p->t_outq.c_cl);

        fprintf(out, " oproc->%8x", p->t_oproc);
        fprintf(out, "  iproc->%8x", p->t_iproc);
        fprintf(out, "  chan->%8x", p->t_chan);
        fprintf(out, "  linep->%8x\n  ", p->t_linep);

        fprintf(out, " addr: %4x", (int)p->t_addr);
        fprintf(out, "  dev: %d/%d", major(p->t_dev), minor(p->t_dev));
        fprintf(out, "  flags: %8x", p->t_flags);
        fprintf(out, "  state: %4x\n  ", (int)p->t_state);

        fprintf(out, " pgrp: %d", (int)p->t_pgrp);
        fprintf(out, "  delct: %d", (int)p->t_delct);
        fprintf(out, "  line: %d", (int)p->t_line);
        fprintf(out, "  col: %d\n  ", (int)p->t_col);
        fprintf(out, " erase: %d", (int)p->t_erase);
        fprintf(out, "  kill: %d", (int)p->t_kill);
        fprintf(out, "  char: %d", (int)p->t_char);
        fprintf(out, "  ispeed: %d", (int)p->t_ispeed);
        fprintf(out, "  ospeed: %d\n  ", (int)p->t_ospeed);

        fprintf(out, " un_cc: %8x", p->t_un.t_ctlq.c_cc);
        fprintf(out, "  un_cf: %8x", p->t_un.t_ctlq.c_cf);
        fprintf(out, "  un_cl: %8x\n  ", p->t_un.t_ctlq.c_cl);

        fprintf(out, " un_intrc: %2x/%c", p->t_un.t_intrc, p->t_un.t_intrc);
        fprintf(out, "  un_quitc: %2x/%c", p->t_un.t_quitc, p->t_un.t_quitc);
        fprintf(out, "  un_startc: %2x/%c\n  ", p->t_un.t_startc, p->t_un.t_startc);

        fprintf(out, " un_stopc: %2x/%c", p->t_un.t_stopc, p->t_un.t_stopc);
        fprintf(out, "  un_eofc: %2x/%c", p->t_un.t_eofc, p->t_un.t_eofc);
        fprintf(out, "  un_brkc: %2x/%c\n  ", p->t_un.t_brkc, p->t_un.t_brkc);
}

/*
 * routine to print the contents of a user structure
 */
prtuser(loc)
int loc;
{
        struct user *p;
        struct proc *pp;
        struct pagetab *tseg, *dseg, *sseg;
        struct segtab *segtab;

        struct saved {
                int sa_grs[16];
                int sa_psw[2];
                double sa_frs[4];
        } *ptr;

        int i, n;

        fprintf(out, "\nUser structure at address %8x", loc);
        n = getbytes(loc, sizeof(int));    /* first word is proc
                                              struct pointer */
        if (n == -1)
                error(1,"illegal address");
        pp = getblk(n, sizeof(struct proc));
        if (pp == -1)
                error(1, "illegal address");
        fprintf(out, " for process %d\n  ", pp -> p_pid); /* rest of hading */

        p = getblk(loc, PSIZE);     /* user structure is one page long */
        if (p == -1)
                error(1, "illegal address");

        fprintf(out, " proc->%8x\n   registers 2-15 on stack exchange\n      ",
              p->u_procp);
        for (i=0; i < 7; i++)
                fprintf(out, "%8x ",p->u_rsav[i]);
        fprintf(out, "\n      ");
        for (i = 7; i < 14; i++)
               fprintf(out, "%8x ", p->u_rsav[i]);

        fprintf(out, "\n   segflg: %d", (int)p->u_segflg);
        fprintf(out, "  error: %d", (int)p->u_error);
        fprintf(out, "  uid: %d", (int)p->u_uid);
        fprintf(out, "  gid: %d", (int)p->u_gid);
        fprintf(out, "  ruid: %d", (int)p->u_ruid);
        fprintf(out, "  rgid: %d\n  ", (int)p->u_rgid);

        fprintf(out, " arglist->%8x\n  ", p->u_ap);

        fprintf(out, " val1(off): %d", p->u_r.r_val1);
        fprintf(out, "  val2(time): %d\n  ", p->u_r.r_val2);

        fprintf(out, " base->%8x", p->u_base);
        fprintf(out, "  count: %d", (int)p->u_count);
        fprintf(out, "  offset: %d\n  ", p->u_offset);

        fprintf(out, " cdir->%8x",p->u_cdir);
        fprintf(out, "  rdir->%8x\n  ",p->u_rdir);

        fprintf(out, " dbuf: %s\n  ", p->u_dbuf);

        fprintf(out, " dirp->%8x\n   current directory entry\n  ",p->u_dirp);
        fprintf(out, "    ino: %d\n  ",p->u_dent.d_ino);
        fprintf(out, "    name: %s\n  ", p->u_dent.d_name);

        fprintf(out, " pdir->%8x\n   open file pointers/flags\n  ", p->u_pdir);
        n = 1;
        for (i = 0; i < NOFILE; i++)
                if (p->u_ofile[i] != NULL) {
                        fprintf(out, "    %d %8x/%2x\n  ",i, p->u_ofile[i],
                                (int)p->u_pofile[i]);
                        n = 0;      /* indicate something done */
                }
        if (n)
                fprintf(out, "    none\n  ");

        fprintf(out, " current args:");
        for (i = 0; i < 4; i++)
                fprintf(out, " %8x", p->u_arg[i]);

        fprintf(out, "\n   registers 2-15 for quits and interrupts\n      ");
        for (i = 0; i < 7; i++)
                fprintf(out, "%8x ", p->u_qsav[i]);
        fprintf(out, "\n      ");
        for (i = 7; i < 14; i++)
                fprintf(out, "%8x ", p->u_qsav[i]);

        fprintf(out, "\n   signals: ");
        for (i = 0; i < NSIG; i++) {
                if (i%8 == 0)
                        fprintf(out, "\n      ");
                fprintf(out, "%d/%d ", i, p->u_signal[i]);
        }

        fprintf(out, "\n   utime: %ld", p->u_utime>>12);
        fprintf(out, "  stime: %ld", p->u_stime>>12);
        fprintf(out, "  cutime: %ld", p->u_cutime>>12);
        fprintf(out, "  cstime: %ld\n  ", p->u_cstime>>12);

        fprintf(out, " ar0->%8x", p->u_ar0);
        fprintf(out, "  intflg: %2x", p->u_intflg);
        fprintf(out, "  sep: %2x\n  ", p->u_sep);

        fprintf(out, " ttyp->%8x", p->u_ttyp);
        fprintf(out, "  ttypd: %d\n  ", p->u_ttyd);

        fprintf(out, " header of executable file\n  ");
        fprintf(out, " magic: %d", p->u_exdata.ux_mag);
        fprintf(out, "  tsize: %d", (int)p->u_exdata.ux_tsize);
        fprintf(out, "  dsize: %d", (int)p->u_exdata.ux_dsize);
        fprintf(out, "  bsize: %d\n  ", (int)p->u_exdata.ux_bsize);
        fprintf(out, " ssize: %d", (int)p->u_exdata.ux_ssize);
        fprintf(out, "  entloc: %8x", (int)p->u_exdata.ux_entloc);
        fprintf(out, "  unused: %d", (int)p->u_exdata.ux_unused);
        fprintf(out, "  relflg: %d\n  ", (int)p->u_exdata.ux_relflg);

        fprintf(out, " comm: %s", p->u_comm);
        fprintf(out, "  start: %d", p->u_start);
        fprintf(out, "  acflag: %2x", (int)p->u_acflag);
        fprintf(out, "  cmask: %4x", (int)p->u_cmask);

        fprintf(out, "\n   program state vector");
        fprintf(out, "(regs and psw from kernel stack in user structure)\n  ");

        ptr = p;
        (int)ptr += ((int)p->u_ar0 - loc);
        fprintf(out, " general registers\n      ");
        for (i = 0; i < 8; i++)
                fprintf(out, "%8x ", ptr->sa_grs[i]);
        fprintf(out, "\n      ");
        for (i = 8; i < 16; i++)
                fprintf(out, "%8x ", ptr->sa_grs[i]);
        fprintf(out, "\n   psw: %8x %8x\n  ", ptr->sa_psw[0],
                ptr->sa_psw[1]);
        fprintf(out, " floating point registers\n      ");
        for (i = 0; i < 4; i++)
                fprintf(out, "%.06e  ", ptr->sa_frs[i]);

        fprintf(out, "\n   sig: %d\n  ", p->u_ps.ps_sig);
        fprintf(out, " cr8-11: %8x %8x %8x %8x\n  ", p->u_ps.ps_cr8,
                p->u_ps.ps_cr9, p->u_ps.ps_cr10, p->u_ps.ps_cr11);
        fprintf(out, " svc: %d", p->u_ps.ps_svc);
        fprintf(out, " pgm: %d", p->u_ps.ps_pgm);
        fprintf(out, " trans: %d", p->u_ps.ps_trans);
        fprintf(out, " moncl: %4x\n  ", p->u_ps.ps_moncl);
        fprintf(out, " per: %d", p->u_ps.ps_per);
        fprintf(out, "  perad: %8x", p->u_ps.ps_perad);
        fprintf(out, "  moncd: %d", p->u_ps.ps_moncd);
        fprintf(out, "  tslice: %ld\n  ", p->u_ps.ps_tslice);

        fprintf(out, " signals: ");
        for (i = 0; i < (NSIG+8)/8; i++)
                fprintf(out, "%2x", (int)p->u_sig[i]);
        fprintf(out, "\n   traced signals: ");
        for (i = 0; i < (NSIG+8)/8; i++)
                fprintf(out, "%2x", (int)p->u_sigtr[i]);

        fprintf(out, "\n   page table pointers/size\n  ");
        fprintf(out, " tseg->%8x/%d\n  ", p->u_tseg.u_pages, p->u_tseg.u_size);
        fprintf(out, " dseg->%8x/%d\n  ", p->u_dseg.u_pages, p->u_dseg.u_size);
        fprintf(out, " sseg->%8x/%d\n  ", p->u_sseg.u_pages, p->u_sseg.u_size);

        fprintf(out, " segtab->%8x\n  ", p->u_segtab);
        fprintf(out, " chreg: %2x", (int)p->u_chreg);
        fprintf(out, "  issig: %2x", (int)p->u_issig);
        fprintf(out, "  rdwr: %d\n  ", p->u_rdwr);
        fprintf(out, " user info: %s\n  ", p->u_info);

        /*
         *  save the page table pointers and dump there contents
         */
        tseg = p->u_tseg.u_pages;
        dseg = p->u_dseg.u_pages;
        sseg = p->u_sseg.u_pages;
        segtab = p-> u_segtab;

        if (tseg != NULL) {
            fprintf(out, " text page tables\n  ");
            prtpagetab(tseg);
        }

        if (dseg != NULL) {
            fprintf(out, " data page tables\n  ");
            prtpagetab(dseg);
        }

        if (sseg != NULL) {
            fprintf(out, " stack page tables\n  ");
            prtpagetab(sseg);
        }

        if (segtab == NULL) return;

        fprintf(out, " segment table (segment number/pointer)\n    ");
        segtab = getblk(segtab, sizeof(union segtab));
        if (segtab == -1)
                error(1, "illegal segtab address");
        n = 0;
        for (i = 0; i <256; i++) {  /* 256 should be SEGLEN if pagetab.h
                                        were up-to-date */
                if (((int)segtab->sg_ptab[i] & 1) == 0) {
                        n++;
                        if (n%6 == 0) fprintf(out, "\n    ");
                        fprintf(out, " %2x/%8x", i, segtab->sg_ptab[i]);
                }
        }
        if (n) fprintf(out, "\n");
}

/*
 * page table print routine
 */
prtpagetab(p)
struct pagetab *p;
{
        struct pagetab *t;
        int i;
        while (p != NULL) {
                p = getblk(p, sizeof(struct pagetab));
                if (p == -1)
                        error(1, "illegal pagetab address");
                t = p->pt_next;
                fprintf(out, "   %8x %8x", t, p->pt_junk);
                for (i = 0; i < PTSIZE; i++) {
                        if (i%8 == 0) fprintf(out, "\n    ");
                        fprintf(out, " %4x", p->pt_entry[i]);
                }
                fprintf(out, "\n  ");
                p = t;
        }
}

/*
 *  routine to print contents of dasd structure
 */
prtdasd(loc)
int loc;
{
        struct dasd *p;

        struct dskdesc *pd;

        int i, j;
        int prtcmddasd();

        fprintf(out, "\nDasd structure at address %8x", loc);

        p = getblk(loc, sizeof(struct dasd));
        if (p == -1)
                error(1, "illegal address");

        fprintf(out, " for dasd %3x\n  ", p->d_devaddr);
        fprintf(out, " channel programs\n  ");
        for (i = 0; i < NCP; i++) {
                fprintf(out, " %3d: ", i);
                prtccw(&p->d_ccws[i][0], prtcmddasd);
                for (j = 1; j < CPSIZE; j++) {
                    fprintf(out, "      ");
                    prtccw(&p->d_ccws[i][j], prtcmddasd);
                }

                fprintf(out, " bb: %d", (int)p->d_recloc[i].d_bb);
                fprintf(out, "  cc: %d", (int)p->d_recloc[i].d_cc);
                fprintf(out, "  hh: %d", (int)p->d_recloc[i].d_hh);
                fprintf(out, "  r0: %d", (int)p->d_recloc[i].d_r0);
                fprintf(out, "  sector: %d\n  ", (int)p->d_recloc[i].d_sector);
        }
        fprintf(out, " cpst: %d", p->d_cpst);
        fprintf(out, "  cpend: %d", p->d_cpend);
        fprintf(out, "  dir: %d", p->d_dir);
        fprintf(out, "  blkno: %d\n  ", p->d_blkno);

        fprintf(out, " iolist->%8x", p->d_iolist);
        fprintf(out, "  desc->%8x", p->d_desc);
        fprintf(out, "  devtab->%8x\n  ", p->d_devtab);

        fprintf(out, " devaddr: %3x", p->d_devaddr);
        fprintf(out, "  maxblk: %d", p->d_maxblk);
        fprintf(out, "  init: %d", p->d_init);
        fprintf(out, "  rw: %d\n  ", p->d_rw);

        fprintf(out, " intreq: %2x", (int)p->d_intreq);
        fprintf(out, "  done: %2x", (int)p->d_done);
        fprintf(out, "  dead: %2x", (int)p->d_dead);
        fprintf(out, "  qlen: %d\n  ", p->d_qlen);

        loc = (int)p->d_desc;
        fprintf(out, " dasd type: ");

        pd = getblk(loc,sizeof(struct dskdesc));
        if (pd == -1)
                error(1, "illegal address");

        switch(pd->d_trkcyl) {
        case 768:
            fprintf(out, "2305-2"); break;
        case 384:
            fprintf(out, "2305-1"); break;
        case 30:
            fprintf(out, "3350"); break;
        case 19:
            fprintf(out, "3330"); break;
        case 12:
            fprintf(out, "3340"); break;
        default:
            fprintf(out, "unknown");
        }
        fprintf(out, "\n");
}

/*
 * Table to translate 3270 code to ASCII
 */
char tubitab[256] = {
        0000, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* 00 */
        0000, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* 08 */
        0000, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* 10 */
        0000, 0000, 0000, 0000, 0000, 0000, 0011, 0000,  /* 18 */
        0000, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* 20 */
        0000, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* 28 */
        0000, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* 30 */
        0000, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* 38 */
        0040, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* 40 */
        0000, 0000, 0134, 0056, 0074, 0050, 0053, 0174,  /* 48 */
        0046, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* 50 */
        0000, 0000, 0041, 0044, 0052, 0051, 0073, 0176,  /* 58 */
        0055, 0057, 0000, 0000, 0000, 0000, 0000, 0000,  /* 60 */
        0000, 0000, 0000, 0054, 0045, 0137, 0076, 0077,  /* 68 */
        0000, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* 70 */
        0000, 0000, 0072, 0043, 0100, 0047, 0075, 0042,  /* 78 */
        0000, 0141, 0142, 0143, 0144, 0145, 0146, 0147,  /* 80 */
        0150, 0151, 0000, 0000, 0000, 0000, 0000, 0000,  /* 88 */
        0000, 0152, 0153, 0154, 0155, 0156, 0157, 0160,  /* 90 */
        0161, 0162, 0000, 0000, 0000, 0000, 0000, 0000,  /* 98 */
        0000, 0000, 0163, 0164, 0165, 0166, 0167, 0170,  /* a0 */
        0171, 0172, 0000, 0000, 0000, 0000, 0000, 0000,  /* a8 */
        0000, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* b0 */
        0000, 0000, 0000, 0000, 0000, 0000, 0000, 0000,  /* b8 */
        0000, 0101, 0102, 0103, 0104, 0105, 0106, 0107,  /* c0 */
        0110, 0111, 0000, 0000, 0000, 0000, 0000, 0000,  /* c8 */
        0000, 0112, 0113, 0114, 0115, 0116, 0117, 0120,  /* d0 */
        0121, 0122, 0000, 0000, 0000, 0000, 0000, 0000,  /* d8 */
        0000, 0000, 0123, 0124, 0125, 0126, 0127, 0130,  /* e0 */
        0131, 0132, 0000, 0000, 0000, 0000, 0000, 0000,  /* e8 */
        0060, 0061, 0062, 0063, 0064, 0065, 0066, 0067,  /* f0 */
        0070, 0071, 0000, 0000, 0000, 0000, 0000, 0000,  /* f8 */
};

/*
 * Routine to print contents of tube structure
 */
prttube(loc)
int loc;
{
        struct tube *p;
        struct tty *t;
        int    *addr;

        int i, j;
        int prtcmdtube();
        char c;

        fprintf(out, "\nTube structure at address %8x", loc);

        addr = (caddr_t)symlook("tptrs");     /* pointers to tube struct */
        t = (caddr_t)symlook("tubes");    /* associated array of tty struct */
        for (i = 0; i < NTUBES; i++) {   /* find tty struct for dev and addr */
             j = getbytes(addr, sizeof(int));
             if (j == loc) {
                  t = getblk(t, sizeof(struct tty));
                  fprintf(out, " for tube %3x, dev %d/%d", t->t_addr,
                                  major(t->t_dev), minor(t->t_dev));
                  break;
             }
             addr++;          /* next pointer in tptrs */
             t++;             /* next structure in tubes */
         }
         fprintf(out, "\n  ");

        p = getblk(loc, sizeof(struct tube));
        if (p == -1)
                error(1, "illegal address");

        fprintf(out, " ccws: ");
        prtccw(&p->t_ccws[0], prtcmdtube);

        fprintf(out, " idaws");
        for (i = 0; i < 32; i++) {
                if (i%8 == 0)
                        fprintf(out, "\n     ");
                fprintf(out, " %8x", p->t_idaws[i]);
        }
        fprintf(out, "\n   curpos: %d", p->t_curpos);
        fprintf(out, "  resid: %d\n  ", p->t_resid);

        fprintf(out, " redisplay back: %d", p->t_re_back);
        fprintf(out, "  forward: %d", p->t_re_forw);
        fprintf(out, "  index: %d", p->t_re_idx);

        fprintf(out, " pksig: %d", p->t_pksig);

        fprintf(out, "  pkproc->%8x\n  ", p->t_pkproc);

        fprintf(out, " saved input buffers:");
        for (i = 0; i < NRE; i++) {
                j = 0;
                fprintf(out, "\n      ");
                while (c = p->t_re_buf[i][j++])
                        fprintf(out, "%c", tubitab[c]);
        }

        fprintf(out, "\n   program function key strings(nulls omitted):");
        for (i = 0; i <= NPFK; i++)
                if ( p->t_pfk[i][0]) {
                        j = 0;
                        fprintf(out, "\n      %d: ", i);
                        while (c = p->t_pfk[i][j++])
                                fprintf(out, "%c", c);
                }

        fprintf(out, "\n  ");

        fprintf(out, "  bits:");
        if (p->t_bits & T_ATTN) fprintf(out, " ATTN");
        if (p->t_bits & T_SKLF) fprintf(out, " SKLF");
        if (p->t_bits & T_NOAT) fprintf(out, " NOAT");
        if (p->t_bits & T_FS)   fprintf(out, " FS");
        if (p->t_bits & T_NCLS) fprintf(out, " NCLS");

        fprintf(out, "\n");
}

/*
 * routine to print mnemonic for tube ccw commands
 */
prtcmdtube(cmd)
int cmd;
{
        switch(cmd) {
        case WRITE:
            fprintf(out, "WRITE"); break;
        case READ:
            fprintf(out, "READ"); break;
        case SENSE:
            fprintf(out, "SENSE"); break;
        case EWRITE:
            fprintf(out, "EWRITE"); break;
        case READMOD:
            fprintf(out, "READMOD"); break;
        case SELECT:
            fprintf(out, "SELECT"); break;
        case WEUA:
            fprintf(out, "WEUA"); break;
        default:
            fprintf(out, "%2x", cmd);
        }
}

/*
 * routine to print mnemonic for dasd ccw commands
 */
prtcmddasd(cmd)
int cmd;
{
        switch(cmd) {
        case SEEK:
            fprintf(out, "SEEK"); break;
        case SETSCTR:
            fprintf(out, "SETSCTR"); break;
        case SRCHIDE:
            fprintf(out, "SRCHIDE"); break;
        case TIC:
            fprintf(out, "TIC"); break;
        case RDDATA:
            fprintf(out, "RDDATA"); break;
        case WRTDATA:
            fprintf(out, "WRTDATA"); break;
        default:
            fprintf(out, "%2x", cmd); break;
        }
}


/*
 * routine to print contents of file structure
 */
prtfile(loc)
int loc;
{
        struct file *p;

        fprintf(out, "\nFile structure at address %8x\n  ", loc);

        p = getblk(loc, sizeof(struct file));
        if (p == -1)
                error(1, "illegal address");

        fprintf(out, " flag: ");
        if (p->f_flag & FREAD)    fprintf(out, "READ ");
        if (p->f_flag & FWRITE)   fprintf(out, "WRITE ");
        if (p->f_flag & FPIPE)    fprintf(out, "PIPE ");
        if (p->f_flag & FMPX)     fprintf(out, "MPX ");
        if (p->f_flag & FMPY)     fprintf(out, "MPY ");
        if (p->f_flag & FMP)      fprintf(out, "MP ");
        if (p->f_flag & FKERNEL)  fprintf(out, "KERNEL ");

        fprintf(out, " count: %d", (int)p->f_count);
        fprintf(out, " inode->%8x\n  ", p->f_inode);

        fprintf(out, " offset: %d or", p->f_offset);
        fprintf(out, " chan->%8x\n", p->f_chan);
}

/*
 * routine to print contents of mount structure
 */
prtmount(loc)
int loc;
{
        struct mount *p;

        fprintf(out, "\nMount structure at address %8x\n  ", loc);

        p = getblk(loc, sizeof(struct mount));
        if (p == -1)
                error(1, "illegal address");

        fprintf(out, " dev: %d/%d", major(p->m_dev), minor(p->m_dev));
        fprintf(out, " bufp->%8x\n  ", p->m_bufp);

        fprintf(out, " inodp->%8x", p->m_inodp);
        fprintf(out, " flag: ");
        if (p->m_flag) fprintf(out, "frozen\n  ");
        else
           fprintf(out, "not frozen\n  ");
}

/*
 * routine to give formatted output of a ccw
 */
prtccw(ccw, cmd)
ccw_t *ccw;
int (*cmd)();
{
        if (ccw->cc_dblw == 0L) {
           fprintf(out, "00000000 00000000\n  ");
           return;
        }
        (*cmd)((int)ccw->cc_cmd);
        fprintf(out, " addr:%6x", (int)ccw->cc_addr);
        fprintf(out, " cd:%d", (int)ccw->cc_cd);
        fprintf(out, " cc:%d", (int)ccw->cc_cc);
        fprintf(out, " sli:%d", (int)ccw->cc_sli);
        fprintf(out, " skip:%d", (int)ccw->cc_skip);
        fprintf(out, " pci:%d", (int)ccw->cc_pci);
        fprintf(out, " ida:%d", (int)ccw->cc_ida);
        fprintf(out, " count:%d\n  ", (int)ccw->cc_count);
}

/*
 * routine to print process command.  Take process address as
 * parameter.
 */
prtcommand(loc)
int loc;          /* process structure location */
{
        struct proc *ptr;

        fprintf(out, "Command for process ");
        ptr = getblk(loc, sizeof(struct proc));
        if (ptr == -1)
                error(1,"illegal address");

        fprintf(out, "%3d: ", (int)ptr->p_pid);

        printcmd(ptr);

}

/*
 *  Routine to find the command, given a process structure.  It has
 *  to get the corresponding user strcutre do do so.
 */
printcmd(ptr)
struct proc *ptr;          /* process structure. */
{
        register int *ip;
        register char *cp, *cp1;
        int c, i, nbad, addr;
	char abuf[512];

        struct user *p;
        struct pagetab *pp;

        p = getblk(ptr -> p_addr,
                   sizeof(struct user));   /* don't need whole thing*/
        if (p == -1)
               error(1, "illegal address");
        if (p->u_sseg.u_pages == NULL) {
               fprintf(out, "\n");
               return(0);
        }
        pp = getblk(p->u_sseg.u_pages, sizeof(struct pagetab));
        if (pp == -1)
	       error(1, "illegal address");
        addr = pp->pt_entry[PTSIZE - 1];  /* determine stack loc */
        addr = (addr >> 4) << 12;
	addr += (4096 - 512);

	cp = getblk(addr, sizeof(abuf));   /* get 512 bytes of stack */
        if (cp == -1)
               error(1, "illegal address");
        for (i = 0; i < 512; i++)
               abuf[i] = *cp++;         /* copy into our buffer */

        /*
         * the rest of the code is copied from /usr/src/cmd/ps.c
         */

	for (ip = (int *)&abuf[512]-2; ip > (int *)abuf; ) {
		if (*--ip == -1 || *ip==0) {
			cp = (char *)(ip+1);
			if (*cp==0)
				cp++;
			nbad = 0;
			for (cp1 = cp; cp1 < &abuf[512]; cp1++) {
				c = *cp1&0177;
				if (c==0)
					*cp1 = ' ';
				else if (c < ' ' || c > 0176) {
					if (++nbad >= 5) {
						*cp1++ = ' ';
						break;
					}
					*cp1 = '?';
				} else if (c=='=') {
					*cp1 = 0;
					while (cp1>cp && *--cp1!=' ')
						*cp1 = 0;
					break;
				}
			}
			while (*--cp1==' ')
				*cp1 = 0;
			fprintf(out, " %.60s\n", cp);
			return(1);
		}
	}
        fprintf(out, "\n");
	return(0);
}

/*
 *  prtstack sets up for a stack traceback, then calls the stack trace
 *  routine.
 */
prtstack(loc)
int     loc;            /* the address of a savearea */
{
        int     *sa
        sa = getblk(loc, 16*4);
        stacktrace(sa);

}

#ifdef DUMP
#define STACKTOP  0xffffff

/*
 * both omak and dcon have their own version of this...
 */
stacktrace(addr)
int     *addr;
{
	register int p, *i, *savereg;
	int lastfunc, *argp, *lastarg;

        fprintf(out,"\nstack traceback\n");
        savereg = addr[12] & STACKTOP;
        lastfunc = addr[10];
	while (savereg != 0) {
                argp = getbytes(savereg+13,4) & STACKTOP;
                lastarg = getbytes(savereg+12,4) & STACKTOP;
                if (lastarg == 0  || lastarg < argp)
                        lastarg = argp+2;
                if (lastarg > argp + 10)  /* in case of bad stack */
                        lastarg = argp+10;
		fprintf(out, "   ");
                symf(lastfunc);
		putc('(', out);
		for (i = argp; i < lastarg; i++){
                        p = getbytes(i,4);
                        fprintf(out, "%x", p);
			if (i != lastarg-1) fprintf(out, ", ");
			}
		fprintf(out, ")\n");
                lastfunc = getbytes(savereg+10,4);
                savereg = getbytes(savereg+12,4) & STACKTOP;
		}
	}
#endif

